/**
  Smart Irrigation System using the Arduino Edge Control - Application Note
  Name: MKR_Cloud_Code.ino
  Purpose: 4 Zones Smart Irrigation System using the Arduino Edge Control with Cloud connectivity using a MKR WiFi 1010.
  @author Christopher Mendez
*/

#include "thingProperties.h"
#include <ArduinoJson.h>
#include <Wire.h>
#include "SensorValues.hpp"
#include <utility/wifi_drv.h>
#include <ArduinoHttpClient.h>
#include <Arduino_JSON.h>

// The MKR1 board I2C address
#define SELF_I2C_ADDR 0x05

// Water tank constants
#define PI 3.1415926535897932384626433832795
#define radius 0.28  // in meters

// I2C communication flow control variables
int ctrlRec = 0;
int ctrlReq = 0;
int valid = 0;

// Valves flow control variables
int memz1 = 0;
int memz2 = 0;
int memz3 = 0;
int memz4 = 0;

// Valves scheduler flow control variables
int schZ1Ctrl = 0;
int schZ2Ctrl = 0;
int schZ3Ctrl = 0;
int schZ4Ctrl = 0;

// Valves countdown variables
int Z1timeCounter = 0;
int Z2timeCounter = 0;
int Z3timeCounter = 0;
int Z4timeCounter = 0;

float currentWater = 0;  // variable to store the initial water level
int waterCtrl = 0;       // flow control for water level
int bootForecast = 1;    // flow control variable for weather variables request

// OpenWeather server address:
const char serverAddress[] = "api.openweathermap.org";
int port = 443;

WiFiSSLClient wifi;
HttpClient client = HttpClient(wifi, serverAddress, port);

// Your OpenWeather API Key
String openWeatherMapApiKey = "<API Key>";

// Replace with your country code and city
String city = "Santiago de los Caballeros";
String countryCode = "DO";

// store last weather variables request
long lastRequest = 0;
// interval between weather variables requests
int interval = 600000;  // 10 mins in seconds

// store last time LED was updated with Cloud connection status
unsigned long previousMillis = 0;
// interval at which to verify connectivity
const long intervalStat = 1000;

/**
  Main section setup
*/
void setup() {

  // Initialize serial and wait for port to open:
  Serial.begin(115200);
  // This delay gives the chance to wait for a Serial Monitor without blocking if none is found
  delay(1500);

  // Defined in thingProperties.h
  initProperties();

  // Connect to Arduino IoT Cloud
  if (!ArduinoCloud.begin(ArduinoIoTPreferredConnection)) {
    Serial.println("ArduinoCloud.begin FAILED!");
  }

  setDebugMessageLevel(2);
  ArduinoCloud.printDebugInfo();

  // Init I2C coomunication
  Wire.begin(SELF_I2C_ADDR);
  Wire.onReceive(receiveEvent);  // I2C receive callback
  Wire.onRequest(requestEvent);  // I2C request callback

  WiFiDrv::pinMode(27, OUTPUT);  //define blue LED pin
}

void loop() {
  // function that ask for schedule timers setted on the Cloud
  scheduleHandler();

  unsigned long currentMillis = millis();

  if (currentMillis - previousMillis >= intervalStat) {
    // save the last time you asked for connectivity status
    previousMillis = currentMillis;

    // Turn on the blue LED if the board is successfully connected to the Cloud.
    if (ArduinoCloud.connected()) {
      WiFiDrv::analogWrite(27, 122);
    } else {
      WiFiDrv::analogWrite(27, 0);
    }
  }

  // doing this just once after there's a WiFi connections
  if (bootForecast == 1 && WiFi.status() == WL_CONNECTED) {
    getForecast();  // request rain probability
    getWeather();   // request temperature and humidity
    bootForecast = 0;
  }


  if (millis() - lastRequest > interval) {
    // request weather variables each interval
    if (WiFi.status() == WL_CONNECTED) {
      getForecast();
      getWeather();
    }
    lastRequest = millis();
  }

  ArduinoCloud.update();
}

/**
  Http GET request to OpenWeather API to retrieve the rain probability 
*/
void getForecast() {
  String path = "/data/2.5/forecast?&cnt=1&q=" + city + "," + countryCode + "&APPID=" + openWeatherMapApiKey;

  // send the GET request
  Serial.println("making GET request");
  client.get(path);

  // read the status code and body of the response
  int statusCode = client.responseStatusCode();
  String response = client.responseBody();
  Serial.print("Status code: ");
  Serial.println(statusCode);

  // parse the string into a JSONVar object:
  JSONVar myObject = JSON.parse(response);

  rain = ((double)myObject["list"][0]["pop"]) * 100.0;
  // if you know the object's keys, you can ask for their values:
  Serial.print("Rain probability:\t");
  Serial.println(rain);
}

/**
  Http GET request to OpenWeather API to retrieve the temperature and humidity
*/
void getWeather() {
  String path = "/data/2.5/weather?q=" + city + "," + countryCode + "&APPID=" + openWeatherMapApiKey;

  // send the GET request
  Serial.println("making GET request");
  client.get(path);

  // read the status code and body of the response
  int statusCode = client.responseStatusCode();
  String response = client.responseBody();
  Serial.print("Status code: ");
  Serial.println(statusCode);

  // parse the string into a JSONVar object:
  JSONVar myObject = JSON.parse(response);

  temperature = (double)myObject["main"]["temp"] - 273.15;  // Kelvin to C
  humidity = (double)myObject["main"]["humidity"];
  // if you know the object's keys, you can ask for their values:
  Serial.print("Temperature:\t");
  Serial.println(temperature);
  Serial.print("Humidity:\t");
  Serial.println(humidity);
}

/*
  Since Z1 is READ_WRITE variable, onZ1Change() is
  executed every time a new value is received from IoT Cloud.
*/
void onZ1Change() {
  // Add your code here to act upon Z1 change
  Serial.println("Z1 changed!!");
  SensorValues_t vals;
  vals.z1_local = z1;
  delay(1000);
}

/*
  Since Z2 is READ_WRITE variable, onZ2Change() is
  executed every time a new value is received from IoT Cloud.
*/
void onZ2Change() {
  // Add your code here to act upon Z2 change
  Serial.println("Z2 changed!!");
  SensorValues_t vals;
  vals.z2_local = z2;
  delay(1000);
}

/*
  Since Z3 is READ_WRITE variable, onZ3Change() is
  executed every time a new value is received from IoT Cloud.
*/
void onZ3Change() {
  // Add your code here to act upon Z3 change
  Serial.println("Z3 changed!!");
  SensorValues_t vals;
  vals.z3_local = z3;
  delay(1000);
}

/*
  Since Z4 is READ_WRITE variable, onZ4Change() is
  executed every time a new value is received from IoT Cloud.
*/
void onZ4Change() {
  // Add your code here to act upon Z4 change
  Serial.println("Z4 changed!!");
  SensorValues_t vals;
  vals.z4_local = z4;
  delay(1000);
}

/**
  Function that updates the Arduino Cloud variables with local values and those received from the Edge Control
*/
void uploadValues(SensorValues_t *vals) {

  StaticJsonDocument<200> doc;
  doc["Zone 1"] = vals->z1_local;
  doc["Zone 2"] = vals->z2_local;
  doc["Zone 3"] = vals->z3_local;
  doc["Zone 4"] = vals->z4_local;
  doc["Timer Zone 1"] = vals->z1_count;
  doc["Timer Zone 2"] = vals->z2_count;
  doc["Timer Zone 3"] = vals->z3_count;
  doc["Timer Zone 4"] = vals->z4_count;
  doc["Water level"] = vals->water_level_local;

  String output;
  serializeJson(doc, output);
  Serial.println(output);

  water_level = vals->water_level_local; // this is in centimeters
  water_volume = PI * (radius) * (radius) * ((int)water_level/100) * 1000;  // we convert water level centimeters in meters resulting on m3 volume, and multiply by 1000 to turn into liters

  if (waterCtrl == 0) {
    currentWater = water_volume;
    Serial.println("Initial Water Volumen: ");
    Serial.println(currentWater);
    water_usage = 0;
    waterCtrl = 1;
  }

  if (water_volume < currentWater) {
    water_usage += currentWater - water_volume;
    currentWater = water_volume;
  } else if (water_volume > currentWater) {
    currentWater = water_volume;
  }

  if (vals->z1_local != memz1) {
    z1 = vals->z1_local;
    memz1 = z1;
  }
  if (vals->z2_local != memz2) {
    z2 = vals->z2_local;
    memz2 = z2;
  }
  if (vals->z3_local != memz3) {
    z3 = vals->z3_local;
    memz3 = z3;
  }
  if (vals->z4_local != memz4) {
    z4 = vals->z4_local;
    memz4 = z4;
  }

  z1_time = vals->z1_on_time;
  z2_time = vals->z2_on_time;
  z3_time = vals->z3_on_time;
  z4_time = vals->z4_on_time;
}


/**
  Function that handles when the Edge Control sends data to the MKR.
  @param bytes The I2C communicated sensors raw values
*/
void receiveEvent(int bytes) {
  uint8_t buf[200];
  uint8_t *ptr = &buf[0];

  SensorValues_t *vals;

  Serial.println("Receive event");
  ctrlRec = 1;
  while (Wire.available() > 0) {
    *ptr = Wire.read();
    ptr++;
  }

  vals = (SensorValues_t *)buf;

  if (ctrlRec - ctrlReq) {
    uploadValues(vals);
  }

  ctrlRec = 0;
  ctrlReq = 0;
}

/**
  Function that handles when the Edge Control request data from the MKR.
*/
void requestEvent() {

  Serial.println("Request event");
  ctrlReq = 1;
  SensorValues_t vals;

  vals.z1_local = z1;
  vals.z2_local = z2;
  vals.z3_local = z3;
  vals.z4_local = z4;

  vals.z1_count = Z1timeCounter;
  vals.z2_count = Z2timeCounter;
  vals.z3_count = Z3timeCounter;
  vals.z4_count = Z4timeCounter;

  uint8_t *buf = (uint8_t *)(&vals);

  for (uint8_t i = 0; i < sizeof(SensorValues_t); i++)
    Wire.write(buf[i]);
}

/**
  Function that handles the scheduled irrigations and ignore them if it will rain
*/
void scheduleHandler() {
  SensorValues_t vals;

  if (rain < 90) {    // if the rain probability is above 90% ignore irrigation
    if (sch_z1.isActive()) {
      if (schZ1Ctrl == 0) {

        Serial.println("Opening Z1 by scheduler");
        vals.z1_local = 1;
        z1 = 1;
        Z1timeCounter = sch_z1.getCloudValue().len;
        Serial.println(Z1timeCounter);
        schZ1Ctrl = 1;
      }

    } else {
      if (schZ1Ctrl == 1) {
        Serial.println("Closing Z1 by scheduler");
        vals.z1_local = 0;
        z1 = 0;
        Z1timeCounter = 0;
        schZ1Ctrl = 0;
      }
    }

    if (sch_z2.isActive()) {
      if (schZ2Ctrl == 0) {
        Serial.println("Opening Z2 by scheduler");
        vals.z2_local = 1;
        z2 = 1;
        Z2timeCounter = sch_z2.getCloudValue().len;
        Serial.println(Z2timeCounter);
        schZ2Ctrl = 1;
      }

    } else {
      if (schZ2Ctrl == 1) {
        Serial.println("Closing Z2 by scheduler");
        vals.z2_local = 0;
        z2 = 0;
        Z2timeCounter = 0;
        schZ2Ctrl = 0;
      }
    }

    if (sch_z3.isActive()) {
      if (schZ3Ctrl == 0) {
        Serial.println("Opening Z3 by scheduler");
        vals.z3_local = 1;
        z3 = 1;
        Z3timeCounter = sch_z3.getCloudValue().len;
        Serial.println(Z3timeCounter);
        schZ3Ctrl = 1;
      }

    } else {
      if (schZ3Ctrl == 1) {
        Serial.println("Closing Z3 by scheduler");
        vals.z3_local = 0;
        z3 = 0;
        Z3timeCounter = 0;
        schZ3Ctrl = 0;
      }
    }

    if (sch_z4.isActive()) {
      if (schZ4Ctrl == 0) {
        Serial.println("Opening Z4 by scheduler");
        vals.z4_local = 1;
        z4 = 1;
        Z4timeCounter = sch_z4.getCloudValue().len;
        Serial.println(Z4timeCounter);
        schZ4Ctrl = 1;
      }

    } else {
      if (schZ4Ctrl == 1) {
        Serial.println("Closing Z4 by scheduler");
        vals.z4_local = 0;
        z4 = 0;
        Z4timeCounter = 0;
        schZ4Ctrl = 0;
      }
    }
  } else {
    Serial.println("Ignoring scheduled irrigation, it will rain!");
  }
}
